﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEditor;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.SceneManagement;
using UnityEngine.UIElements;

public class Player : MonoBehaviour
{
    public static Player instance;

    public GameObject cellphoneP;

    public float moveSpeed = 1f;
    public float alcoholResistance = .75f;
    public float bufferedInputTime = .5f;

    public Animator animator;

    protected bool isActive = true;
    protected int direction = 0;
    protected bool isMoving = false;
    protected float moveStartTime;
    protected Vector3 moveInitialPosition;
    protected Vector3 moveDirection;
    protected List<Tuple<float, int>> bufferedInput = new List<Tuple<float, int>>();

    private void OnEnable()
    {
        instance = this;
        animator = GetComponent<Animator>();
    }

    public void Activate()
    {
        isActive = true;
    }

    public void Deactivate()
    {
        isActive = false;
    }

    private void Start()
    {
        GameController.instance.StartDialog(new string[] {
            "You find yourself in the 37th floor of an apartment building, you see a man you've never seen before. You approach him.",
            "Something weird happens though, you look down... You've moved inside his body. It's like you are in control of his movements, you also share some thoughts.",
            "This is Richard Mills, a pretty tall and muscular fella. It seems like he was a professional cook at some point, but ended up getting involved with the wrong people.",
            "You also sense a strage power within him, something quite familiar to you, what could that be?",
            "INFO: Welcome to Ghost Game, in this game you play the role of a ghost possessing certain individuals while trying to learn more about your own past and what led you to your demise.",
            "INFO: You can move forwards or backwards by pressing the respective keys (Default: W and S). Press the turn keys to make Richard face different directions (Default: A and D).",
            "INFO: Inspect items in the apartment with the interact key (Default: Spacebar).",
            "INFO: While exploring your host may try to fight your commands, if that happens you might feel like you are losing some control over them. You can regain control by completing puzzles or minigames.",
            "INFO: Now go explore the apartment and see if you can learn something about what happened to you o/."
        });
    }

    protected bool Raycast(Vector2 direction, string tag)
    {
        foreach (RaycastHit2D hit in Physics2D.RaycastAll(transform.position, direction, 1f))
        {
            if (hit.transform.CompareTag(tag))
            {
                return true;
            }
        }

        return false;
    }

    protected Vector3 GetFrontVector()
    {
        return direction == 0 ? Vector3.up : direction == 1 ? Vector3.right : direction == 2 ? Vector3.down : Vector3.left;
    }

    protected Vector3 GetBackVector()
    {
        return -GetFrontVector();
    }

    protected void HitUp()
    {
        Vector3 tryMoveDirection = GetFrontVector();
        if (!Raycast(tryMoveDirection, "Obstacle"))
        {
            animator.SetBool("IsMoving", true);
            isMoving = true;
            moveStartTime = Time.time;
            moveInitialPosition = transform.position;
            moveDirection = tryMoveDirection;
        }
    }

    protected void HitDown()
    {
        Vector3 tryMoveDirection = GetBackVector();
        if (!Raycast(tryMoveDirection, "Obstacle"))
        {
            animator.SetBool("IsMoving", true);
            isMoving = true;
            moveStartTime = Time.time;
            moveInitialPosition = transform.position;
            moveDirection = tryMoveDirection;
        }
    }

    protected void HitRight()
    {
        direction = (direction + 1) % 4;
        animator.SetInteger("Direction", direction);
    }

    protected void HitLeft()
    {
        direction = (direction + 3) % 4;
        animator.SetInteger("Direction", direction);
    }

    protected void HitInteract()
    {
        RaycastHit2D[] hits = Physics2D.RaycastAll(transform.position, GetFrontVector(), 1f);
        Array.Sort(hits, new Comparison<RaycastHit2D>((h1, h2) => (int) (10f * (h1.distance - h2.distance))));

        foreach (RaycastHit2D hit in hits)
        {
            Interactable obj = hit.transform.GetComponent<Interactable>();
            if (obj != null && obj.maxRange + .01f >= hit.distance && obj.Interact())
            {
                if (obj.sfx != null)
                {
                    obj.sfx.Play();
                }
                else
                {
                    GameController.instance.PlayInteractSFX();
                }
                return;
            }
        }
    }

    protected void ReadTooltip()
    {
        RaycastHit2D[] hits = Physics2D.RaycastAll(transform.position, GetFrontVector(), 1f);
        Array.Sort(hits, new Comparison<RaycastHit2D>((h1, h2) => (int)(10f * (h1.distance - h2.distance))));

        foreach (RaycastHit2D hit in hits)
        {
            Interactable obj = hit.transform.GetComponent<Interactable>();
            if (obj != null && obj.maxRange + .01f >= hit.distance && obj.tooltip != "")
            {
                TooltipController.instance.Activate(obj.tooltip);
                return;
            }
        }

        TooltipController.instance.Deactivate();
    }

    protected bool DrunkConfusion(int desiredMove)
    {
        if (GameController.instance.HasRestriction("Drunk") && GameController.instance.random.Next(100) > 100f * alcoholResistance)
        {
            while (true)
            {
                int desiredDrunkMove = GameController.instance.random.Next(3);
                if (desiredDrunkMove >= desiredMove)
                {
                    ++desiredDrunkMove;
                }

                switch (desiredDrunkMove)
                {
                    case 0:
                        if (!GameController.instance.HasRestriction("Up"))
                        {
                            Debug.Log("Drunk move: Up");
                            HitUp();
                            return true;
                        }
                        break;
                    case 1:
                        if (!GameController.instance.HasRestriction("Right"))
                        {
                            Debug.Log("Drunk move: Right");
                            HitRight();
                            return true;
                        }
                        break;
                    case 2:
                        if (!GameController.instance.HasRestriction("Down"))
                        {
                            Debug.Log("Drunk move: Down");
                            HitDown();
                            return true;
                        }
                        break;
                    case 3:
                        if (!GameController.instance.HasRestriction("Left"))
                        {
                            Debug.Log("Drunk move: Left");
                            HitLeft();
                            return true;
                        }
                        break;
                }
            }
        }

        return false;
    }

    // Update is called once per frame
    void Update()
    {
        if (!isActive || DialogController.instance.IsActive())
        {
            return;
        }

        if (GameController.instance.HasFlag("Tv"))
        {
            if (Input.GetButtonDown("P") && !GameController.instance.HasFlag("Pickup Phone"))
            {
                cellphoneP.SetActive(false);
                GameController.instance.AddFlag("Pickup Phone");
                GameController.instance.StartDialog(new string[] { "You look at Richard's phone, there's a new message.", "It seems to be talking about you and a job, you recognize the name of the contact.", "You: I've got a new lead, maybe I'll have a better chance of figuring everything out with these new powers.", "You decide to move on to your next lead..." }, () => {
                    SceneManager.LoadScene("End");
                });
            }
            return;
        }

        ReadTooltip();

        if (Input.GetButtonDown("Up"))
        {
            bufferedInput.Add(new Tuple<float, int>(Time.time + bufferedInputTime, 0));
        }

        if (Input.GetButtonDown("Right"))
        {
            bufferedInput.Add(new Tuple<float, int>(Time.time + bufferedInputTime, 1));
        }

        if (Input.GetButtonDown("Down"))
        {
            bufferedInput.Add(new Tuple<float, int>(Time.time + bufferedInputTime, 2));
        }

        if (Input.GetButtonDown("Left"))
        {
            bufferedInput.Add(new Tuple<float, int>(Time.time + bufferedInputTime, 3));
        }

        if (Input.GetButtonDown("Interact"))
        {
            bufferedInput.Add(new Tuple<float, int>(Time.time + bufferedInputTime, 4));
        }

        while (bufferedInput.Count > 2 || bufferedInput.Count > 0 && Time.time > bufferedInput[0].Item1)
        {
            bufferedInput.RemoveAt(0);
        }

        while (!isMoving && bufferedInput.Count > 0)
        {
            switch (bufferedInput[0].Item2)
            {
                case 0:
                    if (!GameController.instance.HasRestriction("Up") && !DrunkConfusion(0))
                    {
                        GameController.instance.IncrementHazardCounter();
                        HitUp();
                    }
                    break;
                case 1:
                    if (!GameController.instance.HasRestriction("Right") && !DrunkConfusion(1))
                    {
                        GameController.instance.IncrementHazardCounter();
                        HitRight();
                    }
                    break;
                case 2:
                    if (!GameController.instance.HasRestriction("Down") && !DrunkConfusion(2))
                    {
                        GameController.instance.IncrementHazardCounter();
                        HitDown();
                    }
                    break;
                case 3:
                    if (!GameController.instance.HasRestriction("Left") && !DrunkConfusion(3))
                    {
                        GameController.instance.IncrementHazardCounter();
                        HitLeft();
                    }
                    break;
                case 4:
                    HitInteract();
                    break;
            }

            bufferedInput.RemoveAt(0);
        }

        if (isMoving)
        {
            float movePercentage = (Time.time - moveStartTime) / moveSpeed;
            if (movePercentage >= 1f)
            {
                GameController.instance.IncrementHazardCounter();
                animator.SetBool("IsMoving", false);
                transform.position = moveInitialPosition + moveDirection;
                isMoving = false;
            }
            else
            {
                transform.position = moveInitialPosition + moveDirection * movePercentage;
            }
        }

        if (Debug.isDebugBuild)
        {
            Debug.DrawRay(transform.position, GetFrontVector(), Color.blue);
            Debug.DrawRay(transform.position, GetBackVector(), Color.red);
        }
    }
}
